from django.forms.models import inlineformset_factory
from django.test import TestCase

from .models import (
    AutoPKChildOfUUIDPKParent,
    AutoPKParent,
    ChildRelatedViaAK,
    ChildWithEditablePK,
    ParentWithUUIDAlternateKey,
    UUIDPKChild,
    UUIDPKChildOfAutoPKParent,
    UUIDPKParent,
)


class InlineFormsetTests(TestCase):
    def test_inlineformset_factory_nulls_default_pks(self):
        """
        #24377 - If we're adding a new object, a parent's auto-generated pk
        from the model field default should be ignored as it's regenerated on
        the save request.

        Tests the case where both the parent and child have a UUID primary key.
        """
        FormSet = inlineformset_factory(UUIDPKParent, UUIDPKChild, fields="__all__")
        formset = FormSet()
        self.assertIsNone(formset.forms[0].fields["parent"].initial)

    def test_inlineformset_factory_ignores_default_pks_on_submit(self):
        """
        #24377 - Inlines with a model field default should ignore that default
        value to avoid triggering validation on empty forms.
        """
        FormSet = inlineformset_factory(UUIDPKParent, UUIDPKChild, fields="__all__")
        formset = FormSet(
            {
                "uuidpkchild_set-TOTAL_FORMS": 3,
                "uuidpkchild_set-INITIAL_FORMS": 0,
                "uuidpkchild_set-MAX_NUM_FORMS": "",
                "uuidpkchild_set-0-name": "Foo",
                "uuidpkchild_set-1-name": "",
                "uuidpkchild_set-2-name": "",
            }
        )
        self.assertTrue(formset.is_valid())
        self.assertIsNone(formset.instance.uuid)
        self.assertIsNone(formset.forms[0].instance.parent_id)

    def test_inlineformset_factory_nulls_default_pks_uuid_parent_auto_child(self):
        """
        #24958 - Variant of test_inlineformset_factory_nulls_default_pks for
        the case of a parent object with a UUID primary key and a child object
        with an AutoField primary key.
        """
        FormSet = inlineformset_factory(
            UUIDPKParent, AutoPKChildOfUUIDPKParent, fields="__all__"
        )
        formset = FormSet()
        self.assertIsNone(formset.forms[0].fields["parent"].initial)

    def test_inlineformset_factory_nulls_default_pks_auto_parent_uuid_child(self):
        """
        #24958 - Variant of test_inlineformset_factory_nulls_default_pks for
        the case of a parent object with an AutoField primary key and a child
        object with a UUID primary key.
        """
        FormSet = inlineformset_factory(
            AutoPKParent, UUIDPKChildOfAutoPKParent, fields="__all__"
        )
        formset = FormSet()
        self.assertIsNone(formset.forms[0].fields["parent"].initial)

    def test_inlineformset_factory_nulls_default_pks_child_editable_pk(self):
        """
        #24958 - Variant of test_inlineformset_factory_nulls_default_pks for
        the case of a parent object with a UUID primary key and a child
        object with an editable natural key for a primary key.
        """
        FormSet = inlineformset_factory(
            UUIDPKParent, ChildWithEditablePK, fields="__all__"
        )
        formset = FormSet()
        self.assertIsNone(formset.forms[0].fields["parent"].initial)

    def test_inlineformset_factory_nulls_default_pks_alternate_key_relation(self):
        """
        #24958 - Variant of test_inlineformset_factory_nulls_default_pks for
        the case of a parent object with a UUID alternate key and a child
        object that relates to that alternate key.
        """
        FormSet = inlineformset_factory(
            ParentWithUUIDAlternateKey, ChildRelatedViaAK, fields="__all__"
        )
        formset = FormSet()
        self.assertIsNone(formset.forms[0].fields["parent"].initial)

    def test_inlineformset_factory_nulls_default_pks_alternate_key_relation_data(self):
        """
        If form data is provided, a parent's auto-generated alternate key is
        set.
        """
        FormSet = inlineformset_factory(
            ParentWithUUIDAlternateKey, ChildRelatedViaAK, fields="__all__"
        )
        formset = FormSet(
            {
                "childrelatedviaak_set-TOTAL_FORMS": 3,
                "childrelatedviaak_set-INITIAL_FORMS": 0,
                "childrelatedviaak_set-MAX_NUM_FORMS": "",
                "childrelatedviaak_set-0-name": "Test",
                "childrelatedviaak_set-1-name": "",
                "childrelatedviaak_set-2-name": "",
            }
        )
        self.assertIs(formset.is_valid(), True)
        self.assertIsNotNone(formset.instance.uuid)
        self.assertEqual(formset.forms[0].instance.parent_id, formset.instance.uuid)
