/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */
#ifndef INCLUDED_SFX2_DINFDLG_HXX
#define INCLUDED_SFX2_DINFDLG_HXX

#include <sal/config.h>
#include <sfx2/dllapi.h>

#include <com/sun/star/util/DateTime.hpp>
#include <com/sun/star/util/Duration.hpp>
#include <com/sun/star/document/CmisProperty.hpp>

#include <svl/stritem.hxx>
#include <svl/zforlist.hxx>

#include <svtools/ctrlbox.hxx>

#include <vcl/idle.hxx>

#include <sfx2/tabdlg.hxx>

#include <boost/optional/optional.hpp>
#include <memory>

namespace com::sun::star::beans { struct PropertyValue; }

namespace com { namespace sun { namespace star {
    namespace document {
        class XDocumentProperties;
    }
} } }

struct CustomProperty;

// class SfxDocumentInfoItem ---------------------------------------------

class SFX2_DLLPUBLIC SfxDocumentInfoItem : public SfxStringItem
{
private:
    sal_Int32                    m_AutoloadDelay;
    OUString                     m_AutoloadURL;
    bool                         m_isAutoloadEnabled;
    OUString                     m_DefaultTarget;
    OUString const               m_TemplateName;
    OUString                     m_Author;
    css::util::DateTime          m_CreationDate;
    OUString                     m_ModifiedBy;
    css::util::DateTime          m_ModificationDate;
    OUString                     m_PrintedBy;
    css::util::DateTime          m_PrintDate;
    sal_Int16                    m_EditingCycles;
    sal_Int32                    m_EditingDuration;
    OUString                     m_Description;
    OUString                     m_Keywords;
    OUString                     m_Subject;
    OUString                     m_Title;
    bool                         m_bHasTemplate;
    bool                         m_bDeleteUserData;
    bool                         m_bUseUserData;
    bool                         m_bUseThumbnailSave;
    std::vector< std::unique_ptr<CustomProperty> >    m_aCustomProperties;
    css::uno::Sequence< css::document::CmisProperty > m_aCmisProperties;

public:
    static SfxPoolItem* CreateDefault();
    SfxDocumentInfoItem();
    SfxDocumentInfoItem( const OUString &rFileName,
        const css::uno::Reference< css::document::XDocumentProperties> & i_xDocProps,
        const css::uno::Sequence< css::document::CmisProperty> & i_cmisProps,
        bool bUseUserData, bool bUseThumbnailSave );
    SfxDocumentInfoItem( const SfxDocumentInfoItem& );
    virtual ~SfxDocumentInfoItem() override;

    /// update i_xDocProps with the data in this object
    void UpdateDocumentInfo(
        const css::uno::Reference< css::document::XDocumentProperties> & i_xDocProps,
        bool i_bDoNotUpdateUserDefined = false)
        const;
    bool        isCmisDocument() const { return m_aCmisProperties.hasElements();}

    bool        isAutoloadEnabled() const { return m_isAutoloadEnabled; }
    void        setAutoloadEnabled(bool i_val) { m_isAutoloadEnabled = i_val; }
    sal_Int32   getAutoloadDelay() const { return m_AutoloadDelay; }
    void        setAutoloadDelay(sal_Int32 i_val) { m_AutoloadDelay = i_val; }
    const OUString& getAutoloadURL() const { return m_AutoloadURL; }
    void        setAutoloadURL(const OUString& i_val) { m_AutoloadURL = i_val; }
    const OUString& getDefaultTarget() const { return m_DefaultTarget; }
    void        setDefaultTarget(const OUString& i_val) { m_DefaultTarget = i_val; }
    const OUString& getTemplateName() const { return m_TemplateName; }
    const OUString& getAuthor() const { return m_Author; }
    void        setAuthor(const OUString& i_val) { m_Author = i_val; }

    const css::util::DateTime&
                getCreationDate() const { return m_CreationDate; }
    const OUString& getModifiedBy() const { return m_ModifiedBy; }
    void        setModifiedBy(const OUString& i_val) { m_ModifiedBy = i_val; }

    const css::util::DateTime&
                getModificationDate() const { return m_ModificationDate; }
    const OUString& getPrintedBy() const { return m_PrintedBy; }
    void        setPrintedBy(const OUString& i_val) { m_PrintedBy = i_val; }
    const css::util::DateTime&
                getPrintDate() const { return m_PrintDate; }
    sal_Int16   getEditingCycles() const { return m_EditingCycles; }
    void        setEditingCycles(sal_Int16 i_val) { m_EditingCycles = i_val; }
    sal_Int32   getEditingDuration() const { return m_EditingDuration; }
    void        setEditingDuration(sal_Int32 i_val) { m_EditingDuration = i_val; }
    const OUString& getDescription() const { return m_Description; }
    void        setDescription(const OUString& i_val) { m_Description = i_val; }
    const OUString& getKeywords() const { return m_Keywords; }
    void        setKeywords(const OUString& i_val) { m_Keywords = i_val; }
    const OUString& getSubject() const { return m_Subject; }
    void        setSubject(const OUString& i_val) { m_Subject = i_val; }
    const OUString& getTitle() const { return m_Title; }
    void        setTitle(const OUString& i_val) { m_Title = i_val; }

    /// reset user-specific data (author, modified-by, ...)
    void        resetUserData(const OUString & i_rAuthor);

    void        SetTemplate( bool b ) { m_bHasTemplate = b; }
    bool        HasTemplate() const { return m_bHasTemplate; }
    void        SetDeleteUserData( bool bSet );
    void        SetUseUserData( bool bSet );
    void        SetUseThumbnailSave( bool bSet );
    bool        IsDeleteUserData() const { return m_bDeleteUserData;}
    bool        IsUseUserData() const { return m_bUseUserData;}
    bool        IsUseThumbnailSave() const { return m_bUseThumbnailSave;}


    std::vector< std::unique_ptr<CustomProperty> > GetCustomProperties() const;
    void        ClearCustomProperties();
    void        AddCustomProperty(  const OUString& sName,
                                    const css::uno::Any& rValue );

    const css::uno::Sequence< css::document::CmisProperty >&
                        GetCmisProperties() const { return m_aCmisProperties;}

    void        SetCmisProperties(const css::uno::Sequence< css::document::CmisProperty >& cmisProps );
    virtual SfxPoolItem*    Clone( SfxItemPool* pPool = nullptr ) const override;
    virtual bool            operator==( const SfxPoolItem& ) const override;
    virtual bool        QueryValue( css::uno::Any& rVal, sal_uInt8 nMemberId = 0 ) const override;
    virtual bool        PutValue( const css::uno::Any& rVal, sal_uInt8 nMemberId ) override;
};

// class SfxDocumentPage -------------------------------------------------

class SfxDocumentPage : public SfxTabPage
{
private:
    OUString                    m_aUnknownSize;
    OUString                    m_aMultiSignedStr;

    bool                        bEnableUseUserData  : 1,
                                bHandleDelete       : 1;

    std::unique_ptr<weld::Image> m_xBmp;
    std::unique_ptr<weld::Label> m_xNameED;
    std::unique_ptr<weld::Button> m_xChangePassBtn;

    std::unique_ptr<weld::Label> m_xShowTypeFT;
    std::unique_ptr<weld::Label> m_xFileValEd;
    std::unique_ptr<weld::Label> m_xShowSizeFT;

    std::unique_ptr<weld::Label> m_xCreateValFt;
    std::unique_ptr<weld::Label> m_xChangeValFt;
    std::unique_ptr<weld::Label> m_xSignedValFt;
    std::unique_ptr<weld::Button> m_xSignatureBtn;
    std::unique_ptr<weld::Label> m_xPrintValFt;
    std::unique_ptr<weld::Label> m_xTimeLogValFt;
    std::unique_ptr<weld::Label> m_xDocNoValFt;

    std::unique_ptr<weld::CheckButton> m_xUseUserDataCB;
    std::unique_ptr<weld::Button> m_xDeleteBtn;
    std::unique_ptr<weld::CheckButton> m_xUseThumbnailSaveCB;

    std::unique_ptr<weld::Label> m_xTemplFt;
    std::unique_ptr<weld::Label> m_xTemplValFt;

    DECL_LINK(DeleteHdl, weld::Button&, void);
    DECL_LINK(SignatureHdl, weld::Button&, void);
    DECL_LINK(ChangePassHdl, weld::Button&, void);
    void                ImplUpdateSignatures();
    void                ImplCheckPasswordState();

protected:
    virtual ~SfxDocumentPage() override;

    virtual bool        FillItemSet( SfxItemSet* ) override;
    virtual void        Reset( const SfxItemSet* ) override;

public:
    SfxDocumentPage(TabPageParent pParent, const SfxItemSet&);
    static VclPtr<SfxTabPage> Create( TabPageParent pParent, const SfxItemSet* );

    void                EnableUseUserData();
};

// class SfxDocumentDescPage ---------------------------------------------

class SfxDocumentDescPage : public SfxTabPage
{
private:
    SfxDocumentInfoItem* m_pInfoItem;
    std::unique_ptr<weld::Entry> m_xTitleEd;
    std::unique_ptr<weld::Entry> m_xThemaEd;
    std::unique_ptr<weld::Entry> m_xKeywordsEd;
    std::unique_ptr<weld::TextView> m_xCommentEd;

protected:
    virtual ~SfxDocumentDescPage() override;

    virtual bool            FillItemSet( SfxItemSet* ) override;
    virtual void            Reset( const SfxItemSet* ) override;

public:
    SfxDocumentDescPage(TabPageParent pParent, const SfxItemSet&);
    static VclPtr<SfxTabPage> Create( TabPageParent pParent, const SfxItemSet* );
};

// class SfxDocumentInfoDialog -------------------------------------------

class SFX2_DLLPUBLIC SfxDocumentInfoDialog : public SfxTabDialogController
{
protected:
    virtual void PageCreated(const OString& rId, SfxTabPage& rPage) override;

public:
    SfxDocumentInfoDialog(weld::Window* pParent, const SfxItemSet&);
    void AddFontTabPage();
};

// class CustomPropertiesRemoveButton ------------------------------------
struct CustomPropertyLine;

class CustomPropertiesDateField
{
private:
    std::unique_ptr<SvtCalendarBox> m_xDateField;
public:
    ::boost::optional<sal_Int16> m_TZ;

    CustomPropertiesDateField(SvtCalendarBox* pDateField);
    void set_visible(bool bVisible) { m_xDateField->set_visible(bVisible); }
    Date get_date() const { return m_xDateField->get_date(); }
    void set_date(const Date& rDate) { m_xDateField->set_date(rDate); }
    ~CustomPropertiesDateField();
};

class CustomPropertiesTimeField
{
public:
    std::unique_ptr<weld::TimeSpinButton> m_xTimeField;
    bool m_isUTC;

    CustomPropertiesTimeField(std::unique_ptr<weld::TimeSpinButton> xTimeField);
    void set_visible(bool bVisible) { m_xTimeField->set_visible(bVisible); }
    tools::Time get_value() const { return m_xTimeField->get_value(); }
    void set_value(const tools::Time& rTime) { m_xTimeField->set_value(rTime); }
    ~CustomPropertiesTimeField();
};

class CustomPropertiesDurationField
{
    css::util::Duration             m_aDuration;
    std::unique_ptr<weld::Entry>    m_xEntry;
    std::unique_ptr<weld::Button>   m_xEditButton;

    DECL_LINK(ClickHdl, weld::Button&, void);
public:
    CustomPropertiesDurationField(std::unique_ptr<weld::Entry> xEntry,
                                  std::unique_ptr<weld::Button> xEditButton);

    void SetDuration( const css::util::Duration& rDuration );
    const css::util::Duration& GetDuration() const { return m_aDuration; }

    void set_visible(bool bVisible);
};

class CustomPropertiesYesNoButton
{
private:
    std::unique_ptr<weld::Widget> m_xTopLevel;
    std::unique_ptr<weld::RadioButton> m_xYesButton;
    std::unique_ptr<weld::RadioButton> m_xNoButton;

public:
    CustomPropertiesYesNoButton(std::unique_ptr<weld::Widget>,
                                std::unique_ptr<weld::RadioButton> xYesButton,
                                std::unique_ptr<weld::RadioButton> xNoButton);
    ~CustomPropertiesYesNoButton();

    void     CheckYes() { m_xYesButton->set_active(true); }
    void     CheckNo() { m_xNoButton->set_active(true); }
    bool     IsYesChecked() const { return m_xYesButton->get_active(); }
    void     set_visible(bool bVisible) { m_xTopLevel->set_visible(bVisible); }
};

class CustomPropertiesWindow;

// struct CustomPropertyLine ---------------------------------------------
struct CustomPropertyLine
{
    CustomPropertiesWindow* m_pParent;

    std::unique_ptr<weld::Builder> m_xBuilder;
    std::unique_ptr<weld::Container> m_xLine;
    std::unique_ptr<weld::ComboBox> m_xNameBox;
    std::unique_ptr<weld::ComboBox> m_xTypeBox;
    std::unique_ptr<weld::Entry> m_xValueEdit;
    std::unique_ptr<weld::Widget> m_xDateTimeBox;
    std::unique_ptr<CustomPropertiesDateField> m_xDateField;
    std::unique_ptr<CustomPropertiesTimeField> m_xTimeField;
    std::unique_ptr<weld::Widget> m_xDurationBox;
    std::unique_ptr<CustomPropertiesDurationField> m_xDurationField;
    std::unique_ptr<CustomPropertiesYesNoButton> m_xYesNoButton;
    std::unique_ptr<weld::Button> m_xRemoveButton;

    bool m_bTypeLostFocus;

    CustomPropertyLine(CustomPropertiesWindow* pParent, weld::Widget* pContainer);
    DECL_LINK(TypeHdl, weld::ComboBox&, void);
    DECL_LINK(RemoveHdl, weld::Button&, void);
    DECL_LINK(EditLoseFocusHdl, weld::Widget&, void);
    DECL_LINK(BoxLoseFocusHdl, weld::Widget&, void);

    void DoTypeHdl(weld::ComboBox& rBox);

    void Clear();
    void Hide();
};

// class CustomPropertiesWindow ------------------------------------------

class CustomPropertiesWindow
{
private:
    sal_Int32                           m_nHeight;
    sal_Int32                           m_nLineHeight;
    sal_Int32                           m_nScrollPos;
    std::vector<std::unique_ptr<CustomProperty>> m_aCustomProperties;
    std::vector<std::unique_ptr<CustomPropertyLine>> m_aCustomPropertiesLines;
    CustomPropertyLine*                 m_pCurrentLine;
    SvNumberFormatter                   m_aNumberFormatter;
    Idle                                m_aEditLoseFocusIdle;
    Idle                                m_aBoxLoseFocusIdle;
    Link<void*,void>                    m_aRemovedHdl;

    weld::Container& m_rBody;
    weld::Label& m_rHeaderAccName;
    weld::Label& m_rHeaderAccType;
    weld::Label& m_rHeaderAccValue;

    DECL_LINK(EditTimeoutHdl, Timer *, void);
    DECL_LINK(BoxTimeoutHdl, Timer *, void);

    bool        IsLineValid( CustomPropertyLine* pLine ) const;
    void        ValidateLine( CustomPropertyLine* pLine, bool bIsFromTypeBox );
    void        CreateNewLine();
    void        FillLine(sal_uInt32 nLine);
    void        StoreCustomProperties();
    sal_uInt32  GetCurrentDataModelPosition() const { return -1 * m_nScrollPos / m_nLineHeight; }

public:
    CustomPropertiesWindow(weld::Container& rParent, weld::Label& rHeaderAccName,
                           weld::Label& rHeaderAccType, weld::Label& rHeaderAccValue);
    ~CustomPropertiesWindow();

    sal_uInt16          GetExistingLineCount() const { return m_aCustomPropertiesLines.size(); }
    sal_uInt16          GetTotalLineCount() const { return m_aCustomProperties.size(); }
    sal_uInt16          GetVisibleLineCount() const;
    void                SetVisibleLineCount(sal_uInt32 nCount);
    sal_Int32           GetHeight() const { return m_nHeight; }
    void                SetHeight(int nHeight) { m_nHeight = nHeight; }
    sal_Int32           GetLineHeight() const { return m_nLineHeight; }
    void                SetLineHeight(sal_Int32 nLineHeight) { m_nLineHeight = nLineHeight; }
    void                AddLine( const OUString& sName, css::uno::Any const & rAny );
    bool                AreAllLinesValid() const;
    void                ClearAllLines();
    void                DoScroll( sal_Int32 nNewPos );
    void                ReloadLinesContent();

    css::uno::Sequence< css::beans::PropertyValue >
                        GetCustomProperties();
    void                SetCustomProperties(std::vector< std::unique_ptr<CustomProperty> >&& rProperties);
    void                SetRemovedHdl( const Link<void*,void>& rLink ) { m_aRemovedHdl = rLink; }

    void                BoxLoseFocus(CustomPropertyLine* pLine);
    void                EditLoseFocus(CustomPropertyLine* pLine);
    void                Remove(CustomPropertyLine* pLine);
};

// class CustomPropertiesControl -----------------------------------------

class CustomPropertiesControl
{
private:
    sal_Int32               m_nThumbPos;

    std::unique_ptr<weld::Widget> m_xBox;
    std::unique_ptr<weld::Container> m_xBody;
    std::unique_ptr<weld::ScrolledWindow> m_xVertScroll;
    std::unique_ptr<CustomPropertiesWindow> m_xPropertiesWin;
    std::unique_ptr<weld::Label> m_xName;
    std::unique_ptr<weld::Label> m_xType;
    std::unique_ptr<weld::Label> m_xValue;

    DECL_LINK( ResizeHdl, const Size&, void );
    DECL_LINK( ScrollHdl, weld::ScrolledWindow&, void );
    DECL_LINK( RemovedHdl, void*, void );

public:
    CustomPropertiesControl();
    ~CustomPropertiesControl();

    void         AddLine(css::uno::Any const & rAny);

    bool         AreAllLinesValid() const { return m_xPropertiesWin->AreAllLinesValid(); }
    void         ClearAllLines() { m_xPropertiesWin->ClearAllLines(); }

    css::uno::Sequence<css::beans::PropertyValue>
                 GetCustomProperties() const
                        { return m_xPropertiesWin->GetCustomProperties(); }
    void         SetCustomProperties(std::vector< std::unique_ptr<CustomProperty> >&& rProperties);

    void         Init(weld::Builder& rParent);
};

// class SfxCustomPropertiesPage -----------------------------------------

class SfxCustomPropertiesPage : public SfxTabPage
{
private:
    DECL_LINK(AddHdl, weld::Button&, void);

    using TabPage::DeactivatePage;

    std::unique_ptr<CustomPropertiesControl> m_xPropertiesCtrl;
    std::unique_ptr<weld::Button> m_xAdd;

protected:
    virtual ~SfxCustomPropertiesPage() override;
    virtual void dispose() override;

    virtual bool        FillItemSet( SfxItemSet* ) override;
    virtual void        Reset( const SfxItemSet* ) override;
    virtual DeactivateRC DeactivatePage( SfxItemSet* pSet ) override;

public:
    SfxCustomPropertiesPage(TabPageParent pParent, const SfxItemSet&);
    static VclPtr<SfxTabPage> Create( TabPageParent pParent, const SfxItemSet* );
};

struct CmisValue
{
    std::unique_ptr<weld::Builder> m_xBuilder;
    std::unique_ptr<weld::Frame>  m_xFrame;
    std::unique_ptr<weld::Entry> m_xValueEdit;

    CmisValue(weld::Widget* pParent, const OUString& rStr);
};

struct CmisDateTime
{
    std::unique_ptr<weld::Builder> m_xBuilder;
    std::unique_ptr<weld::Frame>  m_xFrame;
    std::unique_ptr<SvtCalendarBox>  m_xDateField;
    std::unique_ptr<weld::TimeSpinButton> m_xTimeField;

    CmisDateTime(weld::Widget* pParent, const css::util::DateTime& rDateTime);
};

struct CmisYesNo
{
    std::unique_ptr<weld::Builder> m_xBuilder;
    std::unique_ptr<weld::Frame>  m_xFrame;
    std::unique_ptr<weld::RadioButton> m_xYesButton;
    std::unique_ptr<weld::RadioButton> m_xNoButton;

    CmisYesNo(weld::Widget* pParent, bool bValue);
};

// struct CmisPropertyLine ---------------------------------------------

struct CmisPropertyLine
{
    std::unique_ptr<weld::Builder> m_xBuilder;
    OUString                      m_sId;
    OUString                      m_sType;
    bool                          m_bUpdatable;
    bool                          m_bRequired;
    bool                          m_bMultiValued;
    bool                          m_bOpenChoice;
    std::unique_ptr<weld::Frame>  m_xFrame;
    std::unique_ptr<weld::Label>  m_xName;
    std::unique_ptr<weld::Label>  m_xType;
    std::vector< std::unique_ptr<CmisValue> >     m_aValues;
    std::vector< std::unique_ptr<CmisDateTime> >  m_aDateTimes;
    std::vector< std::unique_ptr<CmisYesNo> >     m_aYesNos;
    long getItemHeight() const;
    CmisPropertyLine(weld::Widget* pParent);
    ~CmisPropertyLine();
};

// class CmisPropertiesWindow ------------------------------------------

class CmisPropertiesWindow
{
private:
    std::unique_ptr<weld::Container>    m_xBox;
    SvNumberFormatter                   m_aNumberFormatter;
    std::vector< std::unique_ptr<CmisPropertyLine> > m_aCmisPropertiesLines;
public:
    CmisPropertiesWindow(std::unique_ptr<weld::Container> xParent);
    ~CmisPropertiesWindow();

    void AddLine( const OUString& sId, const OUString& sName,
                  const OUString& sType, const bool bUpdatable,
                  const bool bRequired, const bool bMultiValued,
                  const bool bOpenChoice,
                  css::uno::Any& aChoices,
                  css::uno::Any const & rAny );
    void ClearAllLines();

    css::uno::Sequence< css::document::CmisProperty >
                        GetCmisProperties() const;
};

// class CmisPropertiesControl -----------------------------------------

class CmisPropertiesControl
{
private:
    CmisPropertiesWindow m_aPropertiesWin;
    std::unique_ptr<weld::ScrolledWindow> m_xScrolledWindow;

public:
    CmisPropertiesControl(weld::Builder& rBuilder);

    void AddLine( const OUString& sId, const OUString& sName,
                  const OUString& sType, const bool bUpdatable,
                  const bool bRequired, const bool bMultiValude,
                  const bool bOpenChoice,
                  css::uno::Any& aChoices,
                  css::uno::Any const & rAny );

    void ClearAllLines();
    css::uno::Sequence< css::document::CmisProperty >
                    GetCmisProperties() const
                        { return m_aPropertiesWin.GetCmisProperties(); }
};

// class SfxCmisPropertiesPage -------------------------------------------------

class SfxCmisPropertiesPage : public SfxTabPage
{
private:
    std::unique_ptr<CmisPropertiesControl> m_xPropertiesCtrl;
    using TabPage::DeactivatePage;

protected:
    virtual bool        FillItemSet( SfxItemSet* ) override;
    virtual void        Reset( const SfxItemSet* ) override;
    virtual DeactivateRC DeactivatePage( SfxItemSet* pSet ) override;

public:
    SfxCmisPropertiesPage(TabPageParent pParent, const SfxItemSet&);
    virtual void dispose() override;
    virtual ~SfxCmisPropertiesPage() override;
    static VclPtr<SfxTabPage> Create( TabPageParent pParent, const SfxItemSet* );
};

#endif // #ifndef _ INCLUDED_SFX2_DINFDLG_HXX

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
