var baseIndexOf = require('../internal/baseIndexOf');

/** Used for native method references. */
var arrayProto = Array.prototype;

/** Native method references. */
var splice = arrayProto.splice;

/**
 * Removes all provided values from `array` using `SameValueZero` for equality
 * comparisons.
 *
 * **Notes:**
 *  - Unlike `_.without`, this method mutates `array`
 *  - [`SameValueZero`](https://people.mozilla.org/~jorendorff/es6-draft.html#sec-samevaluezero)
 *    comparisons are like strict equality comparisons, e.g. `===`, except
 *    that `NaN` matches `NaN`
 *
 * @static
 * @memberOf _
 * @category Array
 * @param {Array} array The array to modify.
 * @param {...*} [values] The values to remove.
 * @returns {Array} Returns `array`.
 * @example
 *
 * var array = [1, 2, 3, 1, 2, 3];
 *
 * _.pull(array, 2, 3);
 * console.log(array);
 * // => [1, 1]
 */
function pull() {
  var args = arguments,
      array = args[0];

  if (!(array && array.length)) {
    return array;
  }
  var index = 0,
      indexOf = baseIndexOf,
      length = args.length;

  while (++index < length) {
    var fromIndex = 0,
        value = args[index];

    while ((fromIndex = indexOf(array, value, fromIndex)) > -1) {
      splice.call(array, fromIndex, 1);
    }
  }
  return array;
}

module.exports = pull;
